import os
from typing import Any, Dict

# Do not remove those imports, even if your IDE says they are unused
import statsmodels
from omegaconf import DictConfig
from pandas import DataFrame
from sklearn.ensemble import GradientBoostingRegressor
from sklearn.linear_model import LassoCV, Lasso
from sklearn.preprocessing import PolynomialFeatures


def save_info_txt(cfg: DictConfig, output_dir: str) -> None:
    """
    Save experiment information to a text file.

    Args:
        cfg (DictConfig): The configuration object containing experiment details.
        output_dir (str): The directory where the text file will be saved.

    Returns:
        None
    """
    experiment_info = {
        "Dataset": str(cfg.dataset),
        "Method": str(cfg.experiment.method_name),
        "Display_name": str(cfg.experiment.display_name),
        "Fully Observable": cfg.fully_observable,
        "Treatment": cfg.treatment,
        "Outcome": cfg.outcome,
        "Seed list": cfg.seeds_list,
        "Success": True,
    }

    with open(os.path.join(output_dir, "experiment_info.txt"), "w") as f:
        for key, value in experiment_info.items():
            f.write(f"{key}: {value}\n")
    print("Experiment complete.")
    return


def print_summary(
    cfg: DictConfig, dataset: str, runtime_summary: DataFrame, results_summary: DataFrame
) -> None:
    """
    Print the summary of the experiment.

    Args:
        cfg (DictConfig): The configuration object containing experiment details.
        dataset (str): The name of the dataset.
        runtime_summary (DataFrame): The summary of the runtime.
        results_summary (DataFrame): The summary of the results.

    Returns:
        None
    """
    print(
        f"Results summary for experiment on {dataset} using {cfg.experiment.method_name}"
    )
    print(results_summary)

    print(
        f"Runtime summary for experiment on {dataset} using {cfg.experiment.method_name}"
    )
    print(runtime_summary)
    return


def dataframe2csv(
    output_dir: str,
    runtime_df: DataFrame,
    results_df: DataFrame,
    runtime_summary: DataFrame,
    results_summary: DataFrame,
) -> None:
    """
    Save dataframes to CSV files.

    Args:
        output_dir (str): The directory where the CSV files will be saved.
        runtime_df (DataFrame): The dataframe containing runtime data.
        results_df (DataFrame): The dataframe containing results data.
        runtime_summary (DataFrame): The summary of the runtime data.
        results_summary (DataFrame): The summary of the results data.

    Returns:
        None
    """
    results_df.to_csv(os.path.join(output_dir, "aggregate_results.csv"), index=False)
    results_summary.to_csv(os.path.join(output_dir, "results_summary.csv"))

    runtime_df.to_csv(os.path.join(output_dir, "aggregate_runtime.csv"), index=False)
    runtime_summary.to_csv(os.path.join(output_dir, "runtime_summary.csv"))
    return


def get_params(experiment_cfg: DictConfig) -> Dict[str, Any]:
    """
    Get the parameters for the experiment.

    Args:
        experiment_cfg (DictConfig): The experiment configuration.

    Returns:
        Dict[str, Any]: The parameters for the experiment.

    Raises:
        ValueError: If method_params in experiment configuration is not a dictionary.
    """
    params = {}
    if "method_params" in experiment_cfg:
        for key in experiment_cfg.method_params:
            match key:
                case "init_params":
                    params["init_params"] = {}
                    # Especially for econml methods, the parameters have to be initialized by running a function
                    for key in experiment_cfg.method_params["init_params"]:
                        params["init_params"][key] = eval(
                            experiment_cfg.method_params["init_params"][key]
                        )
                case "glm_family":
                    # Run python command in "command" and store the output in "params"
                    # This is used because it is possible to recover the parameters for some
                    # methods such as logistic regression just by running a function in DoWhy
                    # NOTE: Install statsmodels with "pip install statsmodels" to use this feature
                    command = experiment_cfg.method_params["glm_family"]
                    params["glm_family"] =  eval(command)
                case _:
                    val = experiment_cfg.method_params[key]
                    params[key] = val if val != "None" else None
            
    else:
        params = {}


    return params


def parse_dict(data_dict: dict[str, Any]) -> dict[str, float]:
    """Parse a dictionary of strings to a dictionary of floats.

    Args:
        data_dict (dict[str, Any]): Dictionary of strings.

    Returns:
        dict[str, float]: Dictionary of floats.
    """

    new_dict = {}
    for key, value in data_dict.items():
        new_dict[key] = float(value)
    return new_dict
